/**
 * server-thread-pool.c
 *
 * Програма моделює роботу багатопотокового паралельного сервера,
 * який здійснює обмін даними з клієнтами через потоковий сокет
 * у комунікаційному домені AF_INET. Сервер прив'язується до всіх
 * IP-адрес свого вузла і до порта, заданого константою SERVER_PORTNUM.
 * Робочі потоки стврюються завчасно. Кожен робочий потік обслуговує
 * одного клієнта; при цьому він у циклі отримує клієнтські повідомлення і 
 * без змін повертає їх назад клієнту (працює як сервер служби "Луна")
 *
 */

#include "myecho.h"

/*
 * Макрос _REENTRANT (див. <features.h>) указує на необхідність
 * використовувати реентерабельні (безпечні в багатопотоковому контексті)
 * варіанти реалізації деяких системних об'єктів, наприклад,
 * змінної errno.
 * Є специфічним для GNU/Linux (в стандарті SUS3 відсутній).
 * Потрібен для забезпечення максимальної сумісності 
 * середовища GNU/Linux з вимогами стандарту SUS3
 * стосовно підтримки багатопотоковості
 */
#define _REENTRANT

#include <assert.h>
#include <errno.h>
#include <netdb.h>
#include <pthread.h>
#include <semaphore.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <unistd.h>

/*
 * Кількість робочих потоків у пулі
 */
#define THREAD_POOL_SIZE	10
/*
 * Журнал помилок
 */
#define ERRLOGFILE "server.err"
/*
 * Дескриптор зв'язаного з (останнім) клієнтом сокета
 */
int conn_s;
/*
 * Семафор, на якому головний потік чекає,
 * поки один із робочих потоків прочитає значення
 * дескриптора з'єднаного з новим клієнтом сокета
 * (значення змінної conn_s)
 */
sem_t conn_sem;
/* 
 * Семафор, на якому робочі потоки чекають встановлення з'єднань
 * з клієнтами
 */
sem_t req_sem;

/*
 * Головна функція робочих потоків.
 * Обслуговує з'єднання з клієнтом
 */
void* handle_connection(void* arg);

/*
 *
 */
int main()
{
	pthread_t thread_id;
	pthread_attr_t thread_attr;
	int s;
	struct sockaddr_in serv_addr;
	int reuseaddr = 1;
	int i;
	int efl;

	// Приєднує стандартний потік виведення повідомлень про помилки
	// до файлу ERRLOGFILE
	if ((freopen(ERRLOGFILE, "w", stderr)) == NULL)
		exit(EXIT_FAILURE);

	// Виконує ініціалізацію семафора conn_sem
	// (присвоюючи йому початкове значення 0)
	if (sem_init(&conn_sem, 0, 0) == -1)
		perror("sem_init()");
	// Виконує ініціалізацію семафора req_sem
	// (присвоюючи йому початкове значення 0)
	if (sem_init(&req_sem, 0, 0) == -1)
		perror("sem_init()");

	// Виконує ініціалізацію контейнера потокових атрибутів
	efl = pthread_attr_init(&thread_attr);
	assert(efl == 0);
	// Атрибуту detachstate присвоює значення PTHREAD_CREATE_DETACHED
	// (створити потік як від'єднаний)
	efl = pthread_attr_setdetachstate(&thread_attr,
	                                  PTHREAD_CREATE_DETACHED);
	assert(efl == 0);
	// Створює робочі потоки
	for (i = 0; i < THREAD_POOL_SIZE; i++) {
		// Ідентифікатори робочих потоків цю програму не цікавлять,
		// тому ідентифікатор кожного потоку вона зберагіє 
		// в тій самій змінній thread_id.
		// Кожен робочий потік виконує функцію handle_connection() і
		// має атрибути у відповідності до контейнера thread_attr 
		efl = pthread_create(&thread_id, &thread_attr,
		                     &handle_connection, NULL);
		if (efl != 0)
			fprintf(stderr, "pthread_create(): %s\n", strerror(efl));
	}

	// Створює потоковий сокет в домені AF_INET
	s = socket(AF_INET, SOCK_STREAM, 0);
	if (s < 0) {
		perror("socket()");
		exit(EXIT_FAILURE);
	}

	// Встановлює для сокета s опцію SO_REUSEADDR
	if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR,
	    (void *) &reuseaddr, (socklen_t) sizeof(int)) != 0) {
		perror("setsockopt()");
		exit(EXIT_FAILURE);
	}

	// Розміщує в структурі serv_addr свою адресу
	memset(&serv_addr, 0, sizeof(serv_addr));
	serv_addr.sin_family = AF_INET;
	serv_addr.sin_addr.s_addr = INADDR_ANY;	// Всі IP-адреси даного вузла
	// htons() переставляє байти свого аргумента згідно з мережевим порядком
	serv_addr.sin_port = htons(SERVER_PORTNUM);
	// Прив'язує сокет s до локальної адреси
	if (bind(s, (struct sockaddr *) &serv_addr, sizeof(serv_addr)) != 0) {
		perror("bind()");
		exit(EXIT_FAILURE);
	}

	// Переводить сокет s в стан готовності до прийому запитів
	// на встановлення з'єднання (і створює для нього чергу запитів
	// на встановлення з'єднання довжиною 5)
	if (listen(s, 5) != 0) {
		perror("listen()");
		exit(EXIT_FAILURE);
	}

    
	while (1) {

		// Встановлює з'єднання з черговим клієнтом
		while (1) {
			conn_s = accept(s, NULL, NULL);
			if (conn_s >= 0)
				break;
			switch (errno) {
			case EINTR:
			case ECONNABORTED:
				continue;
			default:
				perror("accept()");
				exit(EXIT_FAILURE);
			}
		}

		// Додає 1 до семафора req_sem,
		// чим сигналізує робочим потокам про встановлення 
		// з'єднання ще з одним клієнтом
		if (sem_post(&req_sem) != 0)
			perror("sem_post()");
		
		// Чекає, доки один із робочих потоків прочитає
		// значення змінної conn_s
		if (sem_wait(&conn_sem) != 0)
			perror("sem_wait");

	}
}

/*
 * Головна функція робочих потоків.
 * Обслуговує з'єднання з клієнтом.
 * Ніколи не повертає управління
 */
void* handle_connection(void* arg)
{
	while (1) {
		int conn_fd;
		ssize_t n_recved;

		// Переходить у стан чекання,
		// якщо немає з'єджнань з новими клієнтами
		if (sem_wait(&req_sem) != 0)
			perror("sem_wait()");

		// Зберігає в змінній conn_fd дескриптор зв'язаного з клієнтом сокета
		conn_fd = conn_s;
		// Розблоковує головний потік
		if (sem_post(&conn_sem) != 0)
			perror("sem_post()");
    
		// В  циклі отримує й повертає назад клієнтські повідомлення
		while (1) {
			char buf[BUFSIZE];

			n_recved = recv(conn_fd, buf, BUFSIZE, 0);
			if (n_recved > 0) {
				if (send(conn_fd, buf, n_recved, 0) < 0)
					perror("send()");
			} else
				break;
		}
		// Або помилка прийому, або клієнт закрив з'єднання
		if (n_recved < 0)
			perror("recv()");
		if (close(conn_fd) < 0)
			perror("close(conn_fd)");
	}
	return NULL;
}
